#include "SHT4xDrive.hpp"
#include "assets/font/fonts.h"
#ifdef NO_ERROR
#undef NO_ERROR
#endif
#define NO_ERROR 0

static char errorMessage[64];
static int16_t error;

SHT4xDrive::SHT4xDrive(TwoWire &wire, lgfx::LGFX_Device *lcd)
{
    _lcd = lcd;
    _wire = &wire;
}


SHT4xDrive::~SHT4xDrive()
{
    delete _sht4x;
}


void SHT4xDrive::probe()
{
    log_i("Probing SHT4x");
    if (_status == UNIT_STATUS_UNKNOWN) {
        onProbeView();
    }

    _wire->beginTransmission(SHT40_I2C_ADDR_44);
    if (_wire->endTransmission() == 0) {
        // 读取序列号
        uint32_t serialNumber = readSerialNumber();
        if (serialNumber != 0) {
            log_i("Found SHT4x with Serial Number: 0x%08X", serialNumber);
            if (_status == UNIT_STATUS_UNKNOWN) {
                _status = UNIT_STATUS_PROBED;
            }
        }
        return ;
    }

    log_i("SHT4x not found");
    _status = UNIT_STATUS_UNKNOWN;
}


void SHT4xDrive::init()
{
    if (_status == UNIT_STATUS_PROBED) {
        log_i("Initializing SHT4x");
        onInitView();

        _sht4x = new SensirionI2cSht4x();
        _sht4x->begin(*_wire, SHT40_I2C_ADDR_44);
        _sht4x->softReset();
        delay(10);
        _status = UNIT_STATUS_INITIALIZED;
        onInitailizeView();
    }
}

void SHT4xDrive::deinit()
{
    if (_status == UNIT_STATUS_INITIALIZED) {
        log_i("Deinitializing SHT4x");
        probe();
        if (_status == UNIT_STATUS_UNKNOWN) {
            log_i("SHT4x removed");
            onDeinitView();
            delete _sht4x;
        }
    }
}

uint32_t SHT4xDrive::readSerialNumber()
{
    uint8_t data[6];
    uint32_t serialNumber = 0;
    
    // 发送读取序列号命令 0x89
    _wire->beginTransmission(SHT40_I2C_ADDR_44);
    _wire->write(0x89);
    if (_wire->endTransmission() != 0) {
        log_e("Failed to send serial number command");
        return 0;
    }

    // 等待测量完成
    delay(10);

    // 读取 6 字节数据 (2字节数据 + 1字节CRC + 2字节数据 + 1字节CRC)
    _wire->requestFrom(SHT40_I2C_ADDR_44, (uint8_t)6);

    if (_wire->available() == 6) {
        for (int i = 0; i < 6; i++) {
            data[i] = _wire->read();
        }
        
        // 组合序列号 (跳过 CRC 字节: data[2] 和 data[5])
        // 序列号是 32 位: data[0-1] 是高 16 位, data[3-4] 是低 16 位
        serialNumber = ((uint32_t)data[0] << 24) | 
                      ((uint32_t)data[1] << 16) | 
                      ((uint32_t)data[3] << 8) | 
                      ((uint32_t)data[4]);
        
        log_i("SHT4x Serial Number: 0x%08X", serialNumber);
    } else {
        log_e("Failed to read serial number data");
        return 0;
    }

    delay(10);

    return serialNumber;
}


void SHT4xDrive::run()
{
    if (_status != UNIT_STATUS_INITIALIZED) {
        return ;
    }
    log_i("SHT4x onView");

    aTemperature = 0.0;
    aHumidity = 0.0;
    error = _sht4x->measureLowestPrecision(aTemperature, aHumidity);
    if (error != NO_ERROR) {
        errorToString(error, errorMessage, sizeof errorMessage);
        log_e("Error trying to execute measureLowestPrecision(): ", errorMessage);
        return;
    }
    log_i("aTemperature: %f\taHumidity: %f", aTemperature, aHumidity);

    // onInitailizeView();
    onRunView();

    delay(1000);
}


void SHT4xDrive::onProbeView()
{
    _lcd->clear();
    _lcd->startWrite();
    _lcd->drawString("SHT4x", 4, 4, &fonts::efontCN_12_b);
    _lcd->drawCenterString(
        "Probing",
        _lcd->width() / 2,
        _lcd->height() / 2 - _lcd->fontHeight(&fonts::efontCN_24) / 2,
        &fonts::efontCN_24
    );
    _lcd->endWrite();
}


void SHT4xDrive::onInitView()
{
    _lcd->clear();
    _lcd->startWrite();
    _lcd->loadFont(MontserratBold16);
    _lcd->drawString("SHT4x", 4, 4);
    _lcd->drawCenterString(
        "Initializing",
        _lcd->width() / 2,
        _lcd->height() / 2 - _lcd->fontHeight(&fonts::efontCN_24) / 2,
        &fonts::efontCN_24
    );
    _lcd->endWrite();
}


void SHT4xDrive::onInitailizeView()
{
    _lcd->clear();
    _lcd->startWrite();
    _lcd->loadFont(MontserratBold16);
    _lcd->drawString("SHT4x", 4, 4);
    _lcd->loadFont(MontserratSemiBold9);
    _lcd->drawString("(0x44)", 62, 8);

    _lcd->loadFont(MontserratBoldItalic32);
    _lcd->drawString(String(0.00, 0), 4, 28);
    _lcd->drawString(String(0.00, 0), 67, 28);
    _lcd->loadFont(MontserratSemiBold14);
    _lcd->drawString("°C", 48, 41);
    _lcd->drawString("%", 110, 41);

    _lcd->endWrite();
}


void SHT4xDrive::onDeinitView()
{
    _lcd->clear();
    _lcd->startWrite();
    _lcd->drawCenterString(
        "removed",
        _lcd->width() / 2,
        _lcd->height() / 2 - _lcd->fontHeight(&fonts::efontCN_24) / 2,
        &fonts::efontCN_24
    );
    _lcd->endWrite();
    delay(500);
}


void SHT4xDrive::onRunView()
{
    _lcd->startWrite();
    _lcd->loadFont(MontserratBoldItalic32);
    _lcd->drawString(String(aTemperature, 0), 4, 28);
    _lcd->drawString(String(aHumidity, 0), 67, 28);
    _lcd->loadFont(MontserratSemiBold14);
    _lcd->drawString("°C", 46, 41);
    _lcd->drawString("%", 108, 41);
    _lcd->endWrite();
}
